/*
 * WangXun Gigabit PCI Express Linux driver
 * Copyright (c) 2015 - 2017 Beijing WangXun Technology Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * The full GNU General Public License is included in this distribution in
 * the file called "COPYING".
 *
 */

#include "kcompat.h"

#include <linux/pci.h>
#include <linux/if_bridge.h>
#include <linux/pci.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/irq.h>
#include <linux/poll.h>
#include <linux/cdev.h>
#include <linux/device.h>

#define IOMEM __iomem
#define PCI_VENDOR_ID_TRUSTNETIC                0x8088

#define NGBE_FAILED_READ_REG       0xffffffffU

#define DRV_VERSION     __stringify(0.0.2)

static const char wxupgrade_copyright[] =
		"Copyright (c) 2018 -2019 Beijing WangXun Technology Co., Ltd";

#include "command.h"

/* upgrade_pci_tbl - PCI Device ID Table
 *
 * Wildcard entries (PCI_ANY_ID) should come last
 * Last entry must be all 0s
 *
 * { Vendor ID, Device ID, SubVendor ID, SubDevice ID,
 *   Class, Class Mask, private data (not used) }
 */
#ifndef PCI_VDEVICE
#define PCI_VDEVICE(ven, dev)        \
		PCI_VENDOR_ID_##ven, (dev),  \
		PCI_ANY_ID, PCI_ANY_ID, 0, 0
#endif

/* Device IDs */
#define NGBE_DEV_ID_EM_TEST                   0x0000
#define NGBE_DEV_ID_EM_WX1860AL_W             0x0100
#define NGBE_DEV_ID_EM_WX1860A2               0x0101
#define NGBE_DEV_ID_EM_WX1860A2S              0x0102
#define NGBE_DEV_ID_EM_WX1860A4               0x0103
#define NGBE_DEV_ID_EM_WX1860A4S              0x0104
#define NGBE_DEV_ID_EM_WX1860AL2              0x0105
#define NGBE_DEV_ID_EM_WX1860AL2S             0x0106
#define NGBE_DEV_ID_EM_WX1860AL4              0x0107
#define NGBE_DEV_ID_EM_WX1860AL4S             0x0108
#define NGBE_DEV_ID_EM_WX1860NCSI             0x0109
#define NGBE_DEV_ID_EM_WX1860A1               0x010a
#define NGBE_DEV_ID_EM_WX1860A1L              0x010b
#define TXGBE_DEV_ID_SP1000                   0x1001
#define TXGBE_DEV_ID_WX1820                   0x2001
#define TXGBE_DEV_ID_AML5025                  0x5025
#define TXGBE_DEV_ID_AML5125                  0x5125
#define TXGBE_DEV_ID_AML5040                  0x5040
#define TXGBE_DEV_ID_AML5140                  0x5140
#define UPGRADE_CNT   24

//主设备号为0，表示动态分配设备号 
dev_t dev = 0;
static int major = 0;
static int minor = 0;


static struct cdev *upgrade_cdev[UPGRADE_CNT];
static struct class *upgrade_class = NULL;
static struct device * upgrade_dev[UPGRADE_CNT];
static u8 IOMEM *hw_addr[UPGRADE_CNT];

static const struct pci_device_id wxupgrade_pci_tbl[] = {
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_TEST), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860A2), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860A2S), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860A4), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860A4S), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860AL2), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860AL2S), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860AL4), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860AL4S), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860AL_W), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860NCSI), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860A1L), 0},
	{ PCI_VDEVICE(TRUSTNETIC, NGBE_DEV_ID_EM_WX1860A1), 0},
	{ PCI_VDEVICE(TRUSTNETIC, 0x10c), 0},
	{ PCI_VDEVICE(TRUSTNETIC, TXGBE_DEV_ID_SP1000), 0},
	{ PCI_VDEVICE(TRUSTNETIC, TXGBE_DEV_ID_WX1820), 0},
	{ PCI_VDEVICE(TRUSTNETIC, TXGBE_DEV_ID_AML5025), 0},
	{ PCI_VDEVICE(TRUSTNETIC, TXGBE_DEV_ID_AML5125), 0},
	{ PCI_VDEVICE(TRUSTNETIC, TXGBE_DEV_ID_AML5040), 0},
	{ PCI_VDEVICE(TRUSTNETIC, TXGBE_DEV_ID_AML5140), 0},
	/* required last entry */
	{ .device = 0 }
};

int wxupgrade_open(struct inode * pnode, struct file * pfile)
{
	int num = MINOR(pnode->i_rdev);

	if(num >= UPGRADE_CNT)
	{
		return -ENODEV;
	}

	pfile->private_data = hw_addr[num];

    return 0;
}


int wxupgrade_release(struct inode *pnode, struct file *pfile)
{
	return 0;
}

/* read register */
static inline u32
upgrade_rd32(u8 __iomem *base)
{
	return readl(base);
}

static inline u32
rd32(u8 IOMEM *base, u32 reg)
{
	u32 val = NGBE_FAILED_READ_REG;

	if (unlikely(!base))
		return val;

	val = upgrade_rd32(base + reg);

	return val;
}

/* write register */
static inline void
upgrade_wr32(u8 __iomem *base, u32 val)
{
	writel(val, base);
}

static inline void
wr32(u8 IOMEM *base, u32 reg, u32 val)
{
	if (unlikely(!base))
		return;

	upgrade_wr32(base + reg, val);
}

long wxupgrade_ioctl(struct file *pfile, unsigned int cmd, unsigned long arg)
{
	int ret = 0;
	u8 IOMEM *base_addr = pfile->private_data;
	struct upgrade_io *data = (struct upgrade_io *)arg;
	struct upgrade_io addr_val;

	if(!base_addr)
		return -EINVAL;

	switch(cmd) {
	case WXTOOL_READ:
		ret = copy_from_user(&addr_val, data, _IOC_SIZE(cmd));
		addr_val.val = rd32(base_addr, addr_val.addr);
		ret = copy_to_user(data, &addr_val, _IOC_SIZE(cmd));
		break;
	case WXTOOL_WRITE:
		ret = copy_from_user(&addr_val, data, _IOC_SIZE(cmd));
		wr32(base_addr, addr_val.addr, addr_val.val);
		break;
	default:
		printk("unkownd cmd...\n");
		return -EINVAL;
	}

	return ret;
}


//文件操作结构体
static const struct file_operations fops = 
{
	.owner = THIS_MODULE,
	.open = wxupgrade_open,
	.release = wxupgrade_release,
	.unlocked_ioctl = wxupgrade_ioctl,
};


static void setup_cdev(int index)
{
	int err, devno = MKDEV(major, index);

	cdev_init(upgrade_cdev[index], &fops);
	upgrade_cdev[index]->owner = THIS_MODULE;
	upgrade_cdev[index]->ops = &fops;
	err = cdev_add(upgrade_cdev[index], devno, 1);
	if(err)
	{
		printk(KERN_NOTICE "Error %d adding hello%d", err, index);
	}
}

static inline const struct pci_device_id *
pci_match_one_device_wx(const struct pci_device_id *id, const struct pci_dev *dev)
{
	if ((id->vendor == PCI_ANY_ID || id->vendor == dev->vendor) &&
	    (id->device == PCI_ANY_ID || id->device == dev->device) &&
	    (id->subvendor == PCI_ANY_ID || id->subvendor == dev->subsystem_vendor) &&
	    (id->subdevice == PCI_ANY_ID || id->subdevice == dev->subsystem_device) &&
	    !((id->class ^ dev->class) & id->class_mask))
		return id;
	return NULL;
}

static void wxupgrade_map(struct pci_dev *pdev)
{
	int err = 0;	
	char upgrade_name[24];

	if (minor >= UPGRADE_CNT)
		return;

	err = pci_enable_device_mem(pdev);
	if (err)
		return;

	hw_addr[minor] = ioremap(pci_resource_start(pdev, 0),
			  pci_resource_len(pdev, 0));
	if (!hw_addr[minor]) {
		err = -EIO;
		return;
	}

	//构造cdev设备对象
	upgrade_cdev[minor] = cdev_alloc();
	//初始化设备对象
	setup_cdev(minor);
	sprintf(upgrade_name,"wxupgrade_%04x:%02x:%02x.%d", pci_domain_nr(pdev->bus),
					pdev->bus->number,
					PCI_SLOT(pdev->devfn),
					PCI_FUNC(pdev->devfn));

	upgrade_dev[minor] = device_create(upgrade_class, NULL, MKDEV(major, minor), NULL, upgrade_name);
	minor++;
}

static int __init wx_upgrade_init(void)
{
	const struct pci_device_id *ids = wxupgrade_pci_tbl;
	struct pci_dev *pdev = NULL;
	int ret = 0;

	pr_info("%s\n", wxupgrade_copyright);

	//为字符设备动态申请一个设备号
	ret = alloc_chrdev_region(&dev, minor, UPGRADE_CNT, "wxupgrade");
	major = MAJOR(dev);

#ifdef HAVE_CLASS_CREATE_EXTRE_PARAM
	upgrade_class = class_create(THIS_MODULE, "wxupgrade");
#else
	upgrade_class = class_create("wxupgrade");
#endif

	while ((pdev = pci_get_device(PCI_ANY_ID, PCI_ANY_ID, pdev))) {
		ids = wxupgrade_pci_tbl;
		while (ids->vendor || ids->device || ids->class_mask) {
			if (pci_match_one_device_wx(ids, pdev)) {
				wxupgrade_map(pdev);
				break;
			} else {
				ids++;
			}
		}
	}

	return ret;
}

static void __exit wx_upgrade_exit(void)
{
	int i = 0;

	for (i = 0; i < minor; i++) {
		if (hw_addr[minor])
			iounmap(hw_addr[minor]);
	}

	device_destroy(upgrade_class, MKDEV(major, minor));

	for (i = 0; i < minor; i++) {
		device_destroy(upgrade_class, MKDEV(major, i));
	}
	class_destroy(upgrade_class);

	//从内核注销cdev设备对象
	for (i = 0; i < minor; i++) {
		if (upgrade_cdev[i])
			cdev_del(upgrade_cdev[i]);
	}

	//回收设备号
	unregister_chrdev_region(dev, UPGRADE_CNT);
}

MODULE_DEVICE_TABLE(pci, wxupgrade_pci_tbl);
MODULE_AUTHOR("Beijing WangXun Technology Co., Ltd, <linux.nic@trustnetic.com>");
MODULE_DESCRIPTION("WangXun(R) Upgrade Driver");
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VERSION);

module_init(wx_upgrade_init);
module_exit(wx_upgrade_exit);

/* wxupgrade_main.c */

